#include <vector>
#include <iostream>
#include <time.h>
#include <queue>
#include <sstream>
#include <fstream>
#include <string>
#include <set>
#include <stack>
#include <algorithm>
#ifdef _WIN32
#include <windows.h>
#endif

//#include "boost/program_options.hpp"
#include "boost/random.hpp"
#include "boost/random/mersenne_twister.hpp"
#include "boost/random/uniform_int.hpp"
#include "boost/random/uniform_real.hpp"
#include "boost/random/normal_distribution.hpp"

#include "planarmap.h"

using namespace planar;

struct EdgeData {
	int vertexId;
	EdgeData() {};
};

typedef typename Map<EdgeData>::EdgeHandle EdgeHandle;
	
unsigned long seedMix(unsigned long a, unsigned long b, unsigned long c)
{
    a=a-b;  a=a-c;  a=a^(c >> 13);
    b=b-c;  b=b-a;  b=b^(a << 8);
    c=c-a;  c=c-b;  c=c^(b >> 13);
    a=a-b;  a=a-c;  a=a^(c >> 12);
    b=b-c;  b=b-a;  b=b^(a << 16);
    c=c-a;  c=c-b;  c=c^(b >> 5);
    a=a-b;  a=a-c;  a=a^(c >> 3);
    b=b-c;  b=b-a;  b=b^(a << 10);
    c=c-a;  c=c-b;  c=c^(b >> 15);
    return c;
}
unsigned long getseed()
{
#ifdef _WIN32
	unsigned int pid = GetCurrentProcessId(); 
#else
	unsigned int pid = getpid(); 
#endif
	return seedMix(clock(), time(NULL), pid);
}

enum ModelType { SPANNINGTREE, UNIFORM, POLYMER, SCHNYDER };

enum walkSteps { RIGHT=0, LEFT=1, UP=2, DOWN=3 };

template<typename RNG>
std::vector<int> randomDyckPath(int halflength, RNG & rng)
{
	std::vector<int> steps;
	int q=halflength, p=halflength;

	for(int i=0;i<2*halflength;++i)
	{
		boost::random::uniform_int_distribution<> dist(0, (q+p)*(q-p+1)-1);
		if( dist(rng) < (q+1)*(q-p) )
		{
			steps.push_back(-1);
			q--;
		} else
		{
			steps.push_back(1);
			p--;
		}
	}
	return steps;
}

std::vector<int> matchingFromDyckPath(std::vector<int> & path)
{
	std::stack<int> stack;
	std::vector<int> matching;
	for(int i=0,endi=path.size();i<endi;++i)
	{
		if( path[i] == 1 )
		{
			stack.push(i);
			matching.push_back(i);
		}else{
			matching.push_back(stack.top());
			matching[matching.back()] = i;
			stack.pop();
		}
	}
	return matching;
}

template<typename RNG>
std::vector<bool> randomSubsetIndicator(int length, int number, RNG & rng)
{
	std::vector<bool> indicator;
	int rest = number;
	for(int l=length;l>0;--l)
	{
		boost::random::uniform_int_distribution<> dist(1,l);
		if( dist(rng) <= rest )
		{
			indicator.push_back(true);
			rest--;
		} else{
			indicator.push_back(false);
		}
	}
	return indicator;
}

template<typename RNG>
std::vector<walkSteps> randomWalkInQuarterPlane(int halflength, RNG & rng)
{
	// We want to generate a walk in qplane by shuffling two walks on half-line.
	// The precise distribution of the first walk is binom{2n}{2k}*cat(n-k)*cat(k)/(cat(n)*cat(n+1)).
	// To approximate this we generate a normal random variable with mean n/2 and st.dev. sqrt(n/8).
	boost::normal_distribution<> normaldistribution(0.5*halflength, std::sqrt(halflength/8.0) );
	boost::variate_generator<boost::mt19937,boost::normal_distribution<> > variatenormal(rng, normaldistribution);
	int k = static_cast<int>(0.5+variatenormal());
	if( k < 0 ) k=0;
	if( k > halflength ) k=halflength;
	
	std::vector<int> steps1 = randomDyckPath(k,rng);
	std::vector<int> steps2 = randomDyckPath(halflength-k,rng);
	std::vector<bool> indicator = randomSubsetIndicator(2*halflength,2*k,rng);
	
	int num1=0, num2=0;
	std::vector<walkSteps> steps;
	for(int i=0;i<2*halflength;i++)
	{
		if( indicator[i] )
		{
			steps.push_back( steps1[num1] == 1 ? UP : DOWN );
			num1++;
		} else
		{
			steps.push_back( steps2[num2] == 1 ? RIGHT : LEFT );
			num2++;			
		}
	}
	return steps;
}

template<typename RNG>
std::vector<walkSteps> randomDiagonalWalkInQuarterPlane(int halflength, RNG & rng)
{
	std::vector<int> steps1 = randomDyckPath(halflength/2,rng);

	std::vector<walkSteps> steps;
	for(int i=0;i<halflength;i++)
	{
		steps.push_back( steps1[i] == 1 ? UP : DOWN );
		steps.push_back( steps1[i] == 1 ? RIGHT : LEFT );
	}
	return steps;
}

void incrementByStep( walkSteps s, int & x, int & y)
{
	switch( s ) {
		case UP:
			y++;
			break;
		case DOWN:
			y--;
			break;
		case RIGHT:
			x++;
			break;
		case LEFT:
			x--;
	} 
}
walkSteps mirrorStep( walkSteps s)
{
	// RIGHT=0, LEFT=1, UP=2, DOWN=3
	const walkSteps mirror[4] = {UP,DOWN,RIGHT,LEFT};
	return mirror[s];
}

void reflectWalk( std::vector<walkSteps> & walk )
{
	int x=0,y=0;
	
	for(std::vector<walkSteps>::iterator step = walk.begin();step!=walk.end();++step)
	{
		bool abovediagonal = y > x;
		incrementByStep(*step,x,y);
		if( y > x || abovediagonal )
		{
			*step = mirrorStep(*step);
		}
	}
}

void tryMirrorPath(std::vector<std::vector<int> > & path, int pt0, int pt1)
{
	for(int i=pt0+1;i<pt1;++i)
	{
		if( path[pt0][0] - path[i][1] + path[pt0][1] < 0 || path[pt0][1] - path[i][0] + path[pt0][0] < 0 )
			return;
	}
	
	for(int i=pt0+1;i<pt1;++i)
	{
		int tmp = path[pt0][0] - path[i][1] + path[pt0][1];
		path[i][1] = path[pt0][1] - path[i][0] + path[pt0][0];
		path[i][0] = tmp;
	}
}

template<typename RNG>
void markovUpdatePath(std::vector<std::vector<int> > & path, RNG & rng)
{
	int pts[2];
	const int largemovefrequency = 10;
	int smallmovesize = (path.size() < 40 ? path.size()/2 : 20 );
	boost::random::uniform_int_distribution<> moveDist(1,largemovefrequency);
	if( moveDist(rng) == 1 )
	{
		boost::random::uniform_int_distribution<> positionDist(0,path.size()-1);
		pts[0] = positionDist(rng);
		pts[1] = positionDist(rng);
		if( pts[1] < pts[0] ) 
			std::swap( pts[0], pts[1] );
		if( pts[1] <= pts[0] + 1 ) 
			return;
	} else 
	{
		boost::random::uniform_int_distribution<> movesizeDist(2,smallmovesize);
		int movesize = movesizeDist(rng);
		boost::random::uniform_int_distribution<> positionDist(0,path.size()-movesize-1);
		pts[0] = positionDist(rng);
		pts[1] = pts[0] + movesize;
	}
	
	if( path[pts[1]][0]+path[pts[1]][1] == path[pts[0]][0]+path[pts[0]][1] )
	{
		// consider a mirroring operation
		boost::random::uniform_int_distribution<> fiftyfifty(0,1);
		if( fiftyfifty(rng) == 1 )
		{
			tryMirrorPath(path,pts[0],pts[1]);
			return;
		}
	}
		
	int numdiagbefore = 0;
	int numdiagafter = 0;
	for(int i=1;i<pts[1]-pts[0];++i)
	{
		int newcoor[2] = {path[pts[0]][0] + path[pts[1]][0] - path[pts[1]-i][0],
			path[pts[0]][1] + path[pts[1]][1] - path[pts[1]-i][1]};
		if( newcoor[0] < 0 || newcoor[1] < 0 )
			return;
		if( path[pts[0]+i][0] == path[pts[0]+i][1] ) 
			numdiagbefore++;
		if( newcoor[0] == newcoor[1] ) 
			numdiagafter++;
	}

	if( numdiagafter > numdiagbefore )
	{ 
		if( numdiagafter - numdiagbefore > 29 )
			return;
		boost::random::uniform_int_distribution<> dist(1,1<<(numdiagafter - numdiagbefore));
		if( dist(rng) > 1 )
			return;
	}
	for(int i=1,endi=(pts[1]-pts[0])/2;i<=endi;++i)
	{
		for(int dir=0;dir<2;dir++)
		{
			int tmp = path[pts[0]][dir] + path[pts[1]][dir] - path[pts[1]-i][dir];
			path[pts[1]-i][dir] = path[pts[0]][dir] + path[pts[1]][dir] - path[pts[0]+i][dir];
			path[pts[0]+i][dir] = tmp;
		}
	}	
}

std::vector<std::vector<int> > walkToPath(std::vector<walkSteps> & walk)
{
	std::vector<int> x(2,0);
	std::vector<std::vector<int> > path;
	path.push_back(x);
	for(std::vector<walkSteps>::iterator step = walk.begin();step!=walk.end();++step)
	{
		incrementByStep(*step,x[0],x[1]);
		path.push_back(x);
	}
	return path;
}

std::vector<walkSteps> pathToWalk(const std::vector<std::vector<int> > & path)
{
	std::vector<walkSteps> walk;
	for(int i=1,endi=path.size();i<endi;++i)
	{
		if( path[i][0] == path[i-1][0] )
		{
			walk.push_back(path[i][1] > path[i-1][1] ? UP : DOWN);
		} else
		{
			walk.push_back(path[i][0] > path[i-1][0] ? RIGHT : LEFT);			
		}
	}
	return walk;
}

template<typename RNG>
std::vector<walkSteps> randomWalkInCone(int halflength, int steps, RNG & rng)
{
	// produce a walk in 45degree cone, i.e. (0,0) -> (0,0) such that x >= 0 and 0 <= y <= x.
	
	// start with a uniform walk in the quarter plane
	std::vector<walkSteps> walk = randomWalkInQuarterPlane(halflength,rng);
	
	// change to sequence of coordinates
	std::vector<std::vector<int> > path = walkToPath(walk);
	// perform markov steps to bias distribution by (1/2)^(#visits to diagonal)
	for(int i=0;i<steps;++i)
	{
		markovUpdatePath(path,rng);
	}
	// back to sequence of steps
	walk = pathToWalk(path);
	
	// turn it into a walk in cone
	reflectWalk(walk);

	return walk;
}

void walkToMap(const std::vector<walkSteps> & walk, Map<EdgeData> & map )
{
	map.clearMap();
	map.newDoubleEdge();
	map.setRoot(map.getEdge(0));
	EdgeHandle curEdge = map.getRoot()->getNext();
	int size = walk.size();
	for(int i=0;i<size-1;++i)
	{
		switch( walk[i] ) {
			case UP:
				curEdge = map.insertEdge(curEdge->getNext())->getAdjacent();
				break;
			case DOWN:
				curEdge = map.insertEdge(curEdge->getNext(),curEdge->getPrevious(2))->getAdjacent();
				break;
			case RIGHT:
				curEdge = map.insertEdge(curEdge);
				break;
			case LEFT:
				curEdge = map.insertEdge(curEdge,curEdge->getNext(3));
				break;
		} 
	}
}

void outputMapGroup(Map<EdgeData> & map, bool spaceseparated)
{
	if( spaceseparated )
	{
		std::cout << map.numHalfEdges() << "\n";
		for(Map<EdgeData>::EdgeIterator edge = map.begin();edge!=map.end();edge++)
		{
			std::cout << (*edge)->getNext()->getId()+1 
					  << " " << (*edge)->getPrevious()->getId()+1 
					  << " " << (*edge)->getAdjacent()->getId()+1 
					  << "\n";
		}
	} else {
		std::cout << "{";
		for(Map<EdgeData>::EdgeIterator edge = map.begin();edge!=map.end();edge++)
		{
			std::cout << (edge == map.begin() ? "" : ",")
					  << "{" << (*edge)->getNext()->getId()+1 
					  << "," << (*edge)->getPrevious()->getId()+1 
					  << "," << (*edge)->getAdjacent()->getId()+1 
					  << "}";
		}
		std::cout << "}\n";
	}
}

template<typename RNG>
void outputSpanningTreeMap(int size,RNG & rng, bool spaceseparated)
{
	Map<EdgeData> map;
	std::vector<walkSteps> walk = randomWalkInQuarterPlane(size,rng);
	std::vector<int> count(4,0);
	for(int i=0;i<2*size;i++)
	{
		if( walk[i] == UP )	count[0]++;
		if( walk[i] == DOWN )	count[1]++;
		if( walk[i] == RIGHT )	count[2]++;
		if( walk[i] == LEFT )	count[3]++;
	}
	
	walkToMap(walk,map);

	outputMapGroup(map, spaceseparated);
}

template<typename RNG>
void outputPolymerMap(int size,RNG & rng, bool spaceseparated)
{
	Map<EdgeData> map;
	if( size % 2 == 1 ) size++;
	std::vector<walkSteps> walk = randomDiagonalWalkInQuarterPlane(size,rng);
	walkToMap(walk,map);
	
	outputMapGroup(map, spaceseparated);
}

template<typename RNG>
void outputUniformQuadrangulation(int size,RNG & rng, bool spaceseparated)
{

	std::vector<int> dyckpath = randomDyckPath(size,rng);
	std::vector<int> matching = matchingFromDyckPath(dyckpath);
	
	boost::random::uniform_int_distribution<> signdist(-1,1);
	std::vector<int> edgelabels;
	for(int i=0;i<2*size;i++)
	{
		int match = matching[i]; 
		if( match > i )
		{
			edgelabels.push_back(signdist(rng));
		} else {
			edgelabels.push_back(-edgelabels[matching[i]]);
		}
	}
	
	Map<EdgeData> map;
	
	int curLabel = 0;
	std::stack<std::pair<int,EdgeHandle> > stack; 
	std::vector<EdgeHandle> cornerEdges;
	stack.push(std::pair<int,EdgeHandle>(0,map.newDoubleEdge()) );
	cornerEdges.push_back(stack.top().second); 
	for(int i=0;i<2*size-1;++i)
	{
		int nextLabel = curLabel + edgelabels[i];
		if( nextLabel >= curLabel )
		{
			if( matching[i] >= i )
			{
				if( !stack.empty() && stack.top().first == nextLabel )
				{
					stack.top().second = map.insertEdge(stack.top().second->getNext() )->getAdjacent();
				}else{
					stack.push(std::pair<int,EdgeHandle>(nextLabel,map.newDoubleEdge()) );				
				}
			} else
			{
				if( !stack.empty() && stack.top().first == nextLabel )
				{
					stack.top().second = map.insertEdge(cornerEdges[matching[i]],stack.top().second->getNext() );
				}else{
					stack.push(std::pair<int,EdgeHandle>(nextLabel,map.insertEdge(cornerEdges[matching[i]])));			
				}
			}
		} else
		{
			assert(stack.top().first == nextLabel+1 );
			
			EdgeHandle attachEdge = stack.top().second->getNext();
			stack.pop();
			
			if( !stack.empty() && stack.top().first == nextLabel )
			{
				stack.top().second = map.insertEdge(attachEdge,stack.top().second->getNext() );
			}else{
				stack.push(std::pair<int,EdgeHandle>(nextLabel,map.insertEdge(attachEdge) ));				
			}
			
			if( matching[i] < i )
			{
				map.contractVertices( stack.top().second, cornerEdges[matching[i]] );
			} 
		}
		cornerEdges.push_back( stack.top().second );
		
		curLabel += edgelabels[i];
	}
	// Have to contract the vertices that remain in the stack and circle around the root.
	EdgeHandle contractEdge = cornerEdges[0]->getAdjacent()->getPrevious(1-stack.top().first);
	while( stack.size() > 1 )
	{
		map.contractVertices( stack.top().second->getNext(), contractEdge->getNext());
		contractEdge = contractEdge->getPrevious();
		stack.pop();
	}
	
	outputMapGroup(map, spaceseparated);
}

void addFirstSchnyderTree(const std::vector<walkSteps> & walk, Map<EdgeData> & map )
{
	EdgeHandle curEdge = map.getRoot();
	int size = walk.size();
	for(int i=0;i<size;++i)
	{
		switch( walk[i] ) {
			case RIGHT:
			case DOWN:
				// the equivalent of an up-step in Dyck path
				curEdge = map.insertEdge(curEdge)->getNext();
				break;
			case UP:
			case LEFT:
				// the equivalent of a down-step in Dyck path
				curEdge = curEdge->getNext();
				break;
		} 
	}
}

std::vector<EdgeHandle> addSecondSchnyderTree(const std::vector<walkSteps> & walk, Map<EdgeData> & map )
{
	// start with first edge of existing tree
	EdgeHandle curEdge = map.getRoot()->getNext(3);
	int size = walk.size();
	
	std::queue<int> numheads;
	numheads.push(0);
	for(int i=0;i<size;++i)
	{
		if( walk[i] == RIGHT || walk[i] == UP )
		{
			numheads.push(0);
		} else 
		{
			numheads.back()++;
		}
	}

	std::stack<EdgeHandle> tails;
	std::vector<EdgeHandle> greencorners;
	for(int i=1;i<=size+1;++i)
	{
		curEdge = curEdge->getNext();
		
		if( i == size+1 || walk[i-1] == RIGHT || walk[i-1] == DOWN ) 
		{
			// curEdge corresponds to first corner of a vertex
			int heads = numheads.front();
			numheads.pop();
			//assert( heads <= static_cast<int>(tails.size()) );
			while( heads > 0 )
			{
				map.insertEdge(tails.top(),curEdge);
				tails.pop();
				heads--;
			}
			if( i < size )
			{
				greencorners.push_back(curEdge->getPrevious());
			}
		}
		
		if( i < size && (walk[i] == UP || walk[i] == LEFT) )
		{
			// equiv of Dyck down step
			// curEdge corresponds to last corner of a vertex
			tails.push( curEdge );
		}
	}
	//assert( tails.empty() );
	//assert( numheads.empty() );
	return greencorners;
}

template<typename RNG>
void outputSchnyderMap(int size,RNG & rng, bool spaceseparated)
{
	// an ad hoc choice for number of Markov steps, which seems to be sufficient
	int steps = 40 * size;
	if( size % 2 == 1)
	{
		size++;
	}
	size = size/2-1;
	std::vector<walkSteps> walk = randomWalkInCone(size,steps,rng);
	
	Map<EdgeData> map;
	map.makePolygon(3);
	addFirstSchnyderTree(walk,map);
	std::vector<EdgeHandle> greencorners = addSecondSchnyderTree(walk,map);
	for(std::vector<EdgeHandle>::iterator corner = greencorners.begin(); corner != greencorners.end(); ++corner)
	{
		map.insertEdge((*corner)->getNext(),(*corner)->getPrevious() );
	}
	outputMapGroup(map, spaceseparated);
}

int main(int argc, char **argv)
{
	boost::mt19937 rng;
	rng.seed(getseed());

	std::string typeletter = "A";
	int size = 30;
	int number = 1;
	bool spaceseparated = false;

	// parse command line arguments (the naive way)
	for( int i=1;i<argc;i++)
	{
		if( argv[i][0] != '-' )
			continue;
		if( argv[i][1] == 't')
		{
			typeletter = argv[i][2];
		} else if( argv[i][1] == 's' )
		{
			std::istringstream param(argv[i]+2);
			param >> size;
		}else if( argv[i][1] == 'n' )
		{
			std::istringstream param(argv[i]+2);
			param >> number;
		}else if( argv[i][1] == '-' && 0 == strcmp("spaceseparated",argv[i]+2) )
		{
			spaceseparated = true;
		}
	}

	/*
	namespace po = boost::program_options;
	po::options_description desc("Allowed options");
	desc.add_options()
		("help", "produce help message")
		("verbose,v", "print debug information" )
		("type,t", po::value<std::string>(&typeletter)->default_value("A"), "Specifies the model type (A, B, C or D)")
		("size,s", po::value<int>(&size)->default_value(30), "Desired size, i.e. number of faces" )
		("number,n", po::value<int>(&number)->default_value(1), "Number of geometries to be generated" )
		("spaceseparated", "output geometry as a space separated list instead of Mathematica style")
	;	
	po::variables_map vm;
	po::store(po::parse_command_line(argc,argv,desc),vm);
	po::notify(vm);
	
	if( vm.count("help") ) {
		std::cout << desc << "\n";
		return 1;
	}
	
	bool spaceseparated = vm.count("spaceseparated") > 0;
	*/
	
	if( typeletter != "A" && typeletter != "B" && typeletter != "C" && typeletter != "D"  )
	{
		std::cout << "Model type unknown. Select A, B, C or D!\n";
		return 1;
	}
	
	if( number <= 0 )
	{
		std::cout << "The specified number is not a positive integer!\n";
		return 1;
	}
	if( size <= 0 )
	{
		std::cout << "The specified size is not a positive integer!\n";
		return 1;
	}

	ModelType modeltype = SPANNINGTREE;
	if( typeletter == "A" )
	{
		modeltype = SPANNINGTREE;
	} else if( typeletter == "B" )
	{
		modeltype = POLYMER;
	} else if( typeletter == "C" )
	{
		modeltype = UNIFORM;
	} else if( typeletter == "D" )
	{
		modeltype = SCHNYDER;
	}
	
	if( spaceseparated )
		std::cout << number << "\n";
	else
		std::cout << "{";
		
	for(int i=0;i<number;++i)
	{
		if( !spaceseparated && i>0 ) std::cout << ",";
		if( modeltype == SPANNINGTREE )
		{
			outputSpanningTreeMap(size,rng,spaceseparated);
		} else if( modeltype == UNIFORM )
		{
			outputUniformQuadrangulation(size,rng,spaceseparated);
		} else if ( modeltype == POLYMER )
		{
			outputPolymerMap(size,rng,spaceseparated);			
		} else if ( modeltype == SCHNYDER )
		{
			outputSchnyderMap(size,rng,spaceseparated);			
		}
	}
	if( !spaceseparated)
		std::cout << "}\n";
	
	return 0;
}
